package org.datanucleus.store.types;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URI;
import java.net.URL;
import java.util.Collection;
import java.util.Currency;
import java.util.Date;
import java.util.Locale;
import java.util.Set;
import java.util.TimeZone;
import java.util.UUID;

import junit.framework.TestCase;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.JDOClassLoaderResolver;
import org.datanucleus.PersistenceConfiguration;
import org.datanucleus.api.ApiAdapter;
import org.datanucleus.jdo.JDOAdapter;
import org.datanucleus.plugin.PluginManager;
import org.datanucleus.store.types.TypeManager;

/**
 * Component tests for the TypeManager class.
 */
public class TypeManagerTest extends TestCase
{
    TypeManager typeMgr = null;

    /**
     * Start of the test, so log it and initialise.
     * @param name Name of the <tt>TestCase</tt>.
     */
    public TypeManagerTest(String name)
    {
        super(name);
    }

    /**
     * Create a TypeManager for testing.
     */
    protected void setUp() throws Exception
    {
        ApiAdapter api = new JDOAdapter();
        ClassLoaderResolver clr = new JDOClassLoaderResolver();
        PluginManager pluginMgr = new PluginManager(new PersistenceConfiguration() {}, clr);
        pluginMgr.registerExtensionPoints();
        pluginMgr.registerExtensions();
        pluginMgr.resolveConstraints();
        typeMgr = new TypeManager(api, pluginMgr, clr);
    }

    /**
     * Test of the isSupportedSecondClassType() method.
     */
    public void testIsSupportedSecondClassType()
    {
        checkIsSupportedSecondClassType(java.util.Collection.class, true);
        checkIsSupportedSecondClassType(java.util.Set.class, true);
        checkIsSupportedSecondClassType(java.util.HashSet.class, true);
        checkIsSupportedSecondClassType(java.util.Map.class, true);
        checkIsSupportedSecondClassType(java.util.HashMap.class, true);
        checkIsSupportedSecondClassType(java.util.Hashtable.class, true);
        checkIsSupportedSecondClassType(java.util.List.class, true);
        checkIsSupportedSecondClassType(java.util.ArrayList.class, true);
        checkIsSupportedSecondClassType(java.util.LinkedList.class, true);
        checkIsSupportedSecondClassType(java.util.Stack.class, true);
        checkIsSupportedSecondClassType(java.util.Vector.class, true);

        checkIsSupportedSecondClassType(java.util.TreeMap.class, true);
        checkIsSupportedSecondClassType(java.util.TreeSet.class, true);

        checkIsSupportedSecondClassType(boolean.class, true);
        checkIsSupportedSecondClassType(byte.class, true);
        checkIsSupportedSecondClassType(char.class, true);
        checkIsSupportedSecondClassType(double.class, true);
        checkIsSupportedSecondClassType(float.class, true);
        checkIsSupportedSecondClassType(int.class, true);
        checkIsSupportedSecondClassType(long.class, true);
        checkIsSupportedSecondClassType(short.class, true);

        checkIsSupportedSecondClassType(Boolean.class, true);
        checkIsSupportedSecondClassType(Byte.class, true);
        checkIsSupportedSecondClassType(Character.class, true);
        checkIsSupportedSecondClassType(Double.class, true);
        checkIsSupportedSecondClassType(Float.class, true);
        checkIsSupportedSecondClassType(Integer.class, true);
        checkIsSupportedSecondClassType(Long.class, true);
        checkIsSupportedSecondClassType(Short.class, true);

        checkIsSupportedSecondClassType(java.io.File.class, false);
        checkIsSupportedSecondClassType(java.util.Date.class, true);
        checkIsSupportedSecondClassType(java.util.Locale.class, true);
        checkIsSupportedSecondClassType(java.math.BigDecimal.class, true);
        checkIsSupportedSecondClassType(java.math.BigInteger.class, true);
        checkIsSupportedSecondClassType(java.sql.Date.class, true);
        checkIsSupportedSecondClassType(java.sql.Time.class, true);
        checkIsSupportedSecondClassType(java.sql.Timestamp.class, true);

        checkIsSupportedSecondClassType(TestOnlyList.class, true); // Should be true since List is supported
    }

    /**
     * Test of the isSupportedSecondClassType() method for arrays.
     */
    public void testIsSupportedSecondClassTypeForArrays()
    {
        checkIsSupportedArrayType(boolean[].class, true);
        checkIsSupportedArrayType(byte[].class, true);
        checkIsSupportedArrayType(char[].class, true);
        checkIsSupportedArrayType(double[].class, true);
        checkIsSupportedArrayType(float[].class, true);
        checkIsSupportedArrayType(int[].class, true);
        checkIsSupportedArrayType(long[].class, true);
        checkIsSupportedArrayType(short[].class, true);
        checkIsSupportedArrayType(BigInteger[].class, true);
        checkIsSupportedArrayType(BigDecimal[].class, true);
        checkIsSupportedArrayType(String[].class, true);
        checkIsSupportedArrayType(Locale[].class, true);
        checkIsSupportedArrayType(Date[].class, true);

        checkIsSupportedArrayType(java.sql.Date[].class, true); // Should be true since java.util.Date[] is supported
    }

    /**
     * Test for the default value of persistence-modifier.
     * Section 18.13 of JDO2 spec.
     */
    public void testIsDefaultPersistentType()
    {
        // primitives
        checkIsDefaultPersistent(boolean.class, true);
        checkIsDefaultPersistent(byte.class, true);
        checkIsDefaultPersistent(char.class, true);
        checkIsDefaultPersistent(double.class, true);
        checkIsDefaultPersistent(float.class, true);
        checkIsDefaultPersistent(int.class, true);
        checkIsDefaultPersistent(long.class, true);
        checkIsDefaultPersistent(short.class, true);

        // wrappers
        checkIsDefaultPersistent(Boolean.class, true);
        checkIsDefaultPersistent(Byte.class, true);
        checkIsDefaultPersistent(Character.class, true);
        checkIsDefaultPersistent(Double.class, true);
        checkIsDefaultPersistent(Float.class, true);
        checkIsDefaultPersistent(Integer.class, true);
        checkIsDefaultPersistent(Long.class, true);
        checkIsDefaultPersistent(Short.class, true);

        // java.lang
        checkIsDefaultPersistent(String.class, true);
        checkIsDefaultPersistent(Number.class, true);

        // java.math
        checkIsDefaultPersistent(BigDecimal.class, true);
        checkIsDefaultPersistent(BigInteger.class, true);

        // java.util
        checkIsDefaultPersistent(java.util.Currency.class, true);
        checkIsDefaultPersistent(java.util.Locale.class, true);
        checkIsDefaultPersistent(java.util.Date.class, true);
        checkIsDefaultPersistent(java.util.ArrayList.class, true);
        checkIsDefaultPersistent(java.util.HashMap.class, true);
        checkIsDefaultPersistent(java.util.HashSet.class, true);
        checkIsDefaultPersistent(java.util.Hashtable.class, true);
        checkIsDefaultPersistent(java.util.LinkedHashMap.class, true);
        checkIsDefaultPersistent(java.util.LinkedHashSet.class, true);
        checkIsDefaultPersistent(java.util.TreeMap.class, true);
        checkIsDefaultPersistent(java.util.TreeSet.class, true);
        checkIsDefaultPersistent(java.util.LinkedList.class, true);
        checkIsDefaultPersistent(java.util.List.class, true);
        checkIsDefaultPersistent(java.util.Set.class, true);
        checkIsDefaultPersistent(java.util.Map.class, true);

        // Arrays
        checkIsDefaultPersistent(boolean[].class, true);
        checkIsDefaultPersistent(byte[].class, true);
        checkIsDefaultPersistent(char[].class, true);
        checkIsDefaultPersistent(double[].class, true);
        checkIsDefaultPersistent(float[].class, true);
        checkIsDefaultPersistent(int[].class, true);
        checkIsDefaultPersistent(long[].class, true);
        checkIsDefaultPersistent(short[].class, true);
        checkIsDefaultPersistent(Boolean[].class, true);
        checkIsDefaultPersistent(Byte[].class, true);
        checkIsDefaultPersistent(Character[].class, true);
        checkIsDefaultPersistent(Double[].class, true);
        checkIsDefaultPersistent(Float[].class, true);
        checkIsDefaultPersistent(Integer[].class, true);
        checkIsDefaultPersistent(Long[].class, true);
        checkIsDefaultPersistent(Short[].class, true);
        checkIsDefaultPersistent(String[].class, true);
        checkIsDefaultPersistent(Number[].class, true);
        checkIsDefaultPersistent(java.util.Date[].class, true);
        checkIsDefaultPersistent(java.util.Locale[].class, true);
        checkIsDefaultPersistent(BigDecimal[].class, true);
        checkIsDefaultPersistent(BigInteger[].class, true);

        // java.sql
        checkIsDefaultPersistent(java.sql.Date.class, false);
        checkIsDefaultPersistent(java.sql.Time.class, false);
        checkIsDefaultPersistent(java.sql.Timestamp.class, false);
    }

    public void testGetStringConverter()
    {
        checkStringConverterType(BigDecimal.class, BigDecimalStringConverter.class);
        checkStringConverterType(BigInteger.class, BigIntegerStringConverter.class);
        checkStringConverterType(Currency.class, CurrencyStringConverter.class);
        checkStringConverterType(Date.class, DateStringConverter.class);
        checkStringConverterType(Locale.class, LocaleStringConverter.class);
        checkStringConverterType(StringBuffer.class, StringBufferStringConverter.class);
        checkStringConverterType(TimeZone.class, TimeZoneStringConverter.class);
        checkStringConverterType(URI.class, URIStringConverter.class);
        checkStringConverterType(URL.class, URLStringConverter.class);
        checkStringConverterType(UUID.class, UUIDStringConverter.class);

        // null expected if not mappable
        assertNull(typeMgr.getStringConverter(Set.class));
        assertNull(typeMgr.getStringConverter(Object.class));

        // null-check
        // TODO: Is this expected? Or should the method return null?
        try
        {
            typeMgr.getStringConverter(null);
        }
        catch (NullPointerException npe)
        {
            // expected?
        }
    }

    // ------------------------------------ Utility methods --------------------------------

    private void checkIsDefaultPersistent(Class cls, boolean persistent)
    {
        assertEquals("TypeManager default-persistent for " + cls.getName() + " is wrong", 
            typeMgr.isDefaultPersistent(cls), persistent);
    }

    private void checkIsSupportedSecondClassType(Class cls, boolean supported)
    {
        assertTrue("TypeManager claims that support for " + cls.getName() + " is " + supported + " !", 
            supported == typeMgr.isSupportedSecondClassType(cls.getName()));
    }
    
    private void checkIsSupportedArrayType(Class cls, boolean supported)
    {
        assertTrue("TypeManager claims that support for " + cls.getName() + " is " + supported + " !", 
            supported == (cls.isArray() && typeMgr.isSupportedSecondClassType(cls.getName())));
    }

    private void checkStringConverterType(Class type, Class expectedStringConverterType)
    {
        ObjectStringConverter stringConverter = typeMgr.getStringConverter(type);
        assertEquals(expectedStringConverterType, stringConverter.getClass());
    }

    // ------------------------------------ This test class only inner classes --------------------------------

    /**
     * Only used for testing whether subclasses are supported 
     */
    class TestOnlyList extends java.util.ArrayList
    {
        private static final long serialVersionUID = 197823989232L;
        public TestOnlyList()
        {
            super();
        }
        public TestOnlyList(int initialCapacity)
        {
            super(initialCapacity);
        }
        public TestOnlyList(Collection c)
        {
            super(c);
        }
    }
}