package com.google.appengine.api.datastore;

import static com.google.common.base.Preconditions.checkArgument;

import com.google.api.services.datastore.DatastoreV1;
import com.google.api.services.datastore.DatastoreV1.Key;
import com.google.apphosting.datastore.DatastoreV4;
import com.google.apphosting.datastore.DatastoreV4.AllocateIdsRequestOrBuilder;
import com.google.apphosting.datastore.DatastoreV4.AllocateIdsResponse;
import com.google.apphosting.datastore.DatastoreV4.BeginTransactionResponse;
import com.google.apphosting.datastore.DatastoreV4.CommitRequestOrBuilder;
import com.google.apphosting.datastore.DatastoreV4.CommitResponse;
import com.google.apphosting.datastore.DatastoreV4.DeprecatedMutation;
import com.google.apphosting.datastore.DatastoreV4.EntityResult;
import com.google.apphosting.datastore.DatastoreV4.GqlQuery;
import com.google.apphosting.datastore.DatastoreV4.LookupRequestOrBuilder;
import com.google.apphosting.datastore.DatastoreV4.LookupResponse;
import com.google.apphosting.datastore.DatastoreV4.Mutation;
import com.google.apphosting.datastore.DatastoreV4.MutationResult;
import com.google.apphosting.datastore.DatastoreV4.QueryResultBatch;
import com.google.apphosting.datastore.DatastoreV4.ReadOptions;
import com.google.apphosting.datastore.DatastoreV4.RollbackRequestOrBuilder;
import com.google.apphosting.datastore.DatastoreV4.RunQueryRequestOrBuilder;
import com.google.apphosting.datastore.DatastoreV4.RunQueryResponse;
import com.google.apphosting.datastore.EntityV4;
import com.google.apphosting.datastore.EntityV4.PartitionId;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import com.google.protobuf.Parser;

import java.util.Iterator;
import java.util.Set;

/**
 * This class converts corresponding protos between DatastoreV4 and DatastoreV1.
 */
class CloudDatastoreProtoConverter {
  private static final CloudDatastoreProtoConverter INSTANCE =
      new CloudDatastoreProtoConverter();

  static CloudDatastoreProtoConverter getInstance() {
    return INSTANCE;
  }

  private CloudDatastoreProtoConverter() {}

  BeginTransactionResponse.Builder toV4BeginTransactionResponse(
      DatastoreV1.BeginTransactionResponseOrBuilder v1Response) {
    return BeginTransactionResponse.newBuilder()
        .setTransaction(v1Response.getTransaction());
  }

  DatastoreV1.RollbackRequest.Builder toV1RollbackRequest(
      RollbackRequestOrBuilder v4Request) {
    return DatastoreV1.RollbackRequest.newBuilder()
        .setTransaction(v4Request.getTransaction());
  }

  DatastoreV1.RunQueryRequest.Builder toV1RunQueryRequest(
      RunQueryRequestOrBuilder v4Request) {
    DatastoreV1.RunQueryRequest.Builder v1Request = DatastoreV1.RunQueryRequest.newBuilder();
    if (v4Request.hasReadOptions()) {
      v1Request.setReadOptions(toV1ReadOptions(v4Request.getReadOptions()));
    }
    if (v4Request.hasPartitionId()) {
      v1Request.setPartitionId(toV1PartitionId(v4Request.getPartitionId()));
    }
    if (v4Request.hasQuery()) {
      v1Request.setQuery(toV1Query(v4Request.getQuery()));
    }
    if (v4Request.hasGqlQuery()) {
      v1Request.setGqlQuery(toV1GqlQuery(v4Request.getGqlQuery()));
    }
    return v1Request;
  }

  RunQueryResponse.Builder toV4RunQueryResponse(
      DatastoreV1.RunQueryResponseOrBuilder v1Response) {
    return RunQueryResponse.newBuilder()
        .setBatch(toV4QueryResultBatch(v1Response.getBatch()));
  }

  DatastoreV1.LookupRequest.Builder toV1LookupRequest(LookupRequestOrBuilder v4Request) {
    DatastoreV1.LookupRequest.Builder v1Request = DatastoreV1.LookupRequest.newBuilder();
    if (v4Request.hasReadOptions()) {
      v1Request.setReadOptions(toV1ReadOptions(v4Request.getReadOptions()));
    }
    for (EntityV4.Key v4Key : v4Request.getKeyList()) {
      v1Request.addKey(toV1Key(v4Key));
    }
    return v1Request;
  }

  LookupResponse.Builder toV4LookupResponse(DatastoreV1.LookupResponseOrBuilder v1Response) {
    LookupResponse.Builder v4Response = LookupResponse.newBuilder();
    for (DatastoreV1.EntityResult v1FoundResult : v1Response.getFoundList()) {
      v4Response.addFound(toV4EntityResult(v1FoundResult));
    }
    for (DatastoreV1.EntityResult v1MissingResult : v1Response.getMissingList()) {
      v4Response.addMissing(toV4EntityResult(v1MissingResult));
    }
    for (DatastoreV1.Key v1DeferredKey : v1Response.getDeferredList()) {
      v4Response.addDeferred(toV4Key(v1DeferredKey));
    }
    return v4Response;
  }

  DatastoreV1.AllocateIdsRequest.Builder toV1AllocateIdsRequest(
      AllocateIdsRequestOrBuilder v4Request) {
    checkArgument(v4Request.getReserveCount() == 0, "V1 does not support reserve.");
    DatastoreV1.AllocateIdsRequest.Builder v1Request = DatastoreV1.AllocateIdsRequest.newBuilder();
    for (EntityV4.Key v4Key : v4Request.getAllocateList()) {
      v1Request.addKey(toV1Key(v4Key));
    }
    return v1Request;
  }

  AllocateIdsResponse.Builder toV4AllocateIdsResponse(
      DatastoreV1.AllocateIdsResponseOrBuilder v1Response) {
    AllocateIdsResponse.Builder v4Response = AllocateIdsResponse.newBuilder();
    for (DatastoreV1.Key v1Key : v1Response.getKeyList()) {
      v4Response.addAllocated(toV4Key(v1Key));
    }
    return v4Response;
  }

  DatastoreV1.CommitRequest.Builder toV1CommitRequest(CommitRequestOrBuilder v4Request) {
    DatastoreV1.CommitRequest.Builder v1Request = DatastoreV1.CommitRequest.newBuilder();
    if (v4Request.hasTransaction()) {
      v1Request.setTransaction(v4Request.getTransaction());
    }
    checkArgument(!v4Request.hasDeprecatedMutation(), "Deprecated mutations are not supported.");
    if (v4Request.getMutationCount() != 0) {
      DatastoreV1.Mutation.Builder v1Mutations = v1Request.getMutationBuilder();
      for (Mutation v4Mutation : v4Request.getMutationList()) {
        switch (v4Mutation.getOp()) {
          case DELETE:
            if (!v4Mutation.hasKey()) {
              throw new IllegalArgumentException();
            }
            v1Mutations.addDelete(toV1Key(v4Mutation.getKey()));
            break;
          case INSERT:
            throw new IllegalArgumentException();
          case UPDATE:
            if (!v4Mutation.hasEntity()) {
              throw new IllegalArgumentException();
            }
            v1Mutations.addUpdate(toV1Entity(v4Mutation.getEntity()));
            break;
          case UPSERT:
            if (!v4Mutation.hasEntity()) {
              throw new IllegalArgumentException();
            }
            v1Mutations.addUpsert(toV1Entity(v4Mutation.getEntity()));
            break;
          case UNKNOWN:
          default:
            throw new IllegalArgumentException();
        }
      }
    }
    if (v4Request.hasMode()) {
      v1Request.setMode(DatastoreV1.CommitRequest.Mode.valueOf(v4Request.getMode().getNumber()));
    }
    return v1Request;
  }

  CommitResponse.Builder toV4CommitResponse(DatastoreV1.CommitResponseOrBuilder v1Response,
      CommitRequestOrBuilder v4Request) {
    CommitResponse.Builder v4Response = CommitResponse.newBuilder();
    if (v4Request.getMutationCount() != 0) {
      DatastoreV1.MutationResult v1MutationResult = v1Response.getMutationResult();
      Iterator<Key> allocatedIdKeyIt = v1MutationResult.getInsertAutoIdKeyList().iterator();
      for (Mutation v4Mutation : v4Request.getMutationList()) {
        MutationResult.Builder v4MutationResult = v4Response.addMutationResultBuilder();
        if (v4Mutation.getOp() == Mutation.Operation.UPSERT) {
          EntityV4.Key v4RequestKey = v4Mutation.getEntity().getKey();
          EntityV4.Key.PathElement v4ReqPathElt =
              v4RequestKey.getPathElement(v4RequestKey.getPathElementCount() - 1);
          if (!v4ReqPathElt.hasId() && !v4ReqPathElt.hasName()) {
            v4MutationResult.setKey(toV4Key(allocatedIdKeyIt.next()));
          }
        }
      }
      if (allocatedIdKeyIt.hasNext()) {
        throw new IllegalArgumentException("Too many allocated keys.");
      }
      if (v1MutationResult.hasIndexUpdates()) {
        v4Response.setIndexUpdates(v1MutationResult.getIndexUpdates());
      }
    }
    return v4Response;
  }

  static DatastoreV1.Mutation toV1Mutation(DeprecatedMutation v4DeprecatedMutation) {
    return checkedProtoConversion(DatastoreV1.Mutation.PARSER, v4DeprecatedMutation);
  }

  static MutationResult toV4MutationResult(DatastoreV1.MutationResult v1MutationResult) {
    return checkedProtoConversion(MutationResult.PARSER, v1MutationResult);
  }

  static QueryResultBatch toV4QueryResultBatch(DatastoreV1.QueryResultBatch v1QueryResultBatch) {
    return checkedProtoConversion(DatastoreV4.QueryResultBatch.PARSER, v1QueryResultBatch);
  }

  static DatastoreV1.PartitionId toV1PartitionId(PartitionId v4PartitionId) {
    return checkedProtoConversion(DatastoreV1.PartitionId.PARSER, v4PartitionId);
  }

  static DatastoreV1.ReadOptions toV1ReadOptions(ReadOptions v4ReadOptions) {
    return checkedProtoConversion(DatastoreV1.ReadOptions.PARSER, v4ReadOptions);
  }

  static DatastoreV1.GqlQuery toV1GqlQuery(GqlQuery v4GqlQuery) {
    return checkedProtoConversion(DatastoreV1.GqlQuery.PARSER, v4GqlQuery);
  }

  static DatastoreV1.Query toV1Query(DatastoreV4.Query v4Query) {
    return checkedProtoConversion(DatastoreV1.Query.PARSER, v4Query);
  }

  static DatastoreV1.Entity toV1Entity(EntityV4.Entity v4Entity) {
    return checkedProtoConversion(DatastoreV1.Entity.PARSER, v4Entity);
  }

  static EntityResult toV4EntityResult(DatastoreV1.EntityResult v1EntityResult) {
    return checkedProtoConversion(EntityResult.PARSER, v1EntityResult);
  }

  static DatastoreV1.Key toV1Key(EntityV4.Key v4Key) {
    return checkedProtoConversion(DatastoreV1.Key.PARSER, v4Key);
  }

  static EntityV4.Key toV4Key(DatastoreV1.Key v1Key) {
    return checkedProtoConversion(EntityV4.Key.PARSER, v1Key);
  }

  private static <T extends Message> T checkedProtoConversion(Parser<T> parser, Message source) {
    try {
      T result = parser.parseFrom(source.toByteArray());
      Set<String> unknownFields = new UnknownFieldChecker().getUnknownFields(result);
      if (!unknownFields.isEmpty()) {
        throw new IllegalArgumentException("Conversion from [" + source + "] has unknown fields:\n"
            + unknownFields);
      }
      return result;
    } catch (InvalidProtocolBufferException e) {
      throw new RuntimeException(e);
    }
  }
}
