package com.google.appengine.api.datastore;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

import com.google.api.client.auth.oauth2.Credential;
import com.google.api.client.googleapis.auth.oauth2.GoogleCredential;
import com.google.api.client.googleapis.compute.ComputeCredential;
import com.google.api.client.googleapis.javanet.GoogleNetHttpTransport;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpRequestInitializer;
import com.google.api.client.http.javanet.NetHttpTransport;
import com.google.api.client.json.jackson.JacksonFactory;
import com.google.api.services.datastore.DatastoreV1;
import com.google.api.services.datastore.client.Datastore;
import com.google.api.services.datastore.client.DatastoreException;
import com.google.api.services.datastore.client.DatastoreFactory;
import com.google.api.services.datastore.client.DatastoreOptions;
import com.google.appengine.api.datastore.DatastoreServiceConfig.ApiVersion;
import com.google.apphosting.api.ApiProxy;
import com.google.apphosting.api.ApiProxy.ApiConfig;
import com.google.apphosting.api.ApiProxy.ApiProxyException;
import com.google.apphosting.api.ApiProxy.Delegate;
import com.google.apphosting.api.ApiProxy.Environment;
import com.google.apphosting.api.ApiProxy.EnvironmentFactory;
import com.google.apphosting.api.ApiProxy.LogRecord;
import com.google.apphosting.datastore.DatastoreV4.AllocateIdsRequest;
import com.google.apphosting.datastore.DatastoreV4.AllocateIdsResponse;
import com.google.apphosting.datastore.DatastoreV4.BeginTransactionRequest;
import com.google.apphosting.datastore.DatastoreV4.BeginTransactionResponse;
import com.google.apphosting.datastore.DatastoreV4.CommitRequest;
import com.google.apphosting.datastore.DatastoreV4.CommitResponse;
import com.google.apphosting.datastore.DatastoreV4.ContinueQueryRequest;
import com.google.apphosting.datastore.DatastoreV4.ContinueQueryResponse;
import com.google.apphosting.datastore.DatastoreV4.LookupRequest;
import com.google.apphosting.datastore.DatastoreV4.LookupResponse;
import com.google.apphosting.datastore.DatastoreV4.RollbackRequest;
import com.google.apphosting.datastore.DatastoreV4.RollbackResponse;
import com.google.apphosting.datastore.DatastoreV4.RunQueryRequest;
import com.google.apphosting.datastore.DatastoreV4.RunQueryResponse;
import com.google.protobuf.Message;

import java.io.File;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletResponse;

/**
 * Redirects API calls to Google Cloud Datastore.
 */
final class CloudDatastoreProxy implements DatastoreV4Proxy {

  private static final ExecutorService executor = Executors.newCachedThreadPool();

  private static final CloudDatastoreProtoConverter CONVERTER =
      CloudDatastoreProtoConverter.getInstance();

  private final Datastore datastore;

  CloudDatastoreProxy(Datastore datastore) {
    this.datastore = checkNotNull(datastore);
  }

  /**
   * Creates a {@link CloudDatastoreProxy}. This method has the side effect
   * of installing minimal stubs ({@link EnvironmentFactory} and
   * {@link Delegate}) in the API proxy if they have not already been installed.
   */
  static CloudDatastoreProxy create(DatastoreServiceConfig config) {
    checkArgument(config.getApiVersion() == ApiVersion.CLOUD_DATASTORE);
    DatastoreOptions options;
    try {
      options = getDatastoreOptions();
    } catch (GeneralSecurityException | IOException e) {
      throw new RuntimeException(
          "Could not get Cloud Datastore options from environment.", e);
    }
    ensureApiProxyIsConfigured(options);
    return new CloudDatastoreProxy(DatastoreFactory.get().create(options));
  }

  @Override
  public Future<BeginTransactionResponse> beginTransaction(BeginTransactionRequest v4Request) {
    return makeCall(new Callable<BeginTransactionResponse>() {
      @Override
      public BeginTransactionResponse call() throws DatastoreException {
        return CONVERTER.toV4BeginTransactionResponse(datastore.beginTransaction(
            DatastoreV1.BeginTransactionRequest.getDefaultInstance())).build();
      }
    });
  }

  @Override
  public Future<RollbackResponse> rollback(final RollbackRequest v4Request) {
    return makeCall(new Callable<RollbackResponse>() {
      @Override
      public RollbackResponse call() throws DatastoreException {
        datastore.rollback(CONVERTER.toV1RollbackRequest(v4Request).build());
        return RollbackResponse.getDefaultInstance();
      }
    });
  }

  @Override
  public Future<RunQueryResponse> runQuery(final RunQueryRequest v4Request) {
    return makeCall(new Callable<RunQueryResponse>() {
      @Override
      public RunQueryResponse call() throws DatastoreException {
        return CONVERTER.toV4RunQueryResponse(datastore.runQuery(
            CONVERTER.toV1RunQueryRequest(v4Request).build())).build();
      }
    });
  }

  @Override
  public Future<ContinueQueryResponse> continueQuery(ContinueQueryRequest v4Request) {
    throw new UnsupportedOperationException();
  }

  @Override
  public Future<LookupResponse> lookup(final LookupRequest v4Request) {
    return makeCall(new Callable<LookupResponse>() {
      @Override
      public LookupResponse call() throws DatastoreException {
        return CONVERTER.toV4LookupResponse(datastore.lookup(
            CONVERTER.toV1LookupRequest(v4Request).build())).build();
      }
    });
  }

  @Override
  public Future<AllocateIdsResponse> allocateIds(final AllocateIdsRequest v4Request) {
    return makeCall(new Callable<AllocateIdsResponse>() {
      @Override
      public AllocateIdsResponse call() throws DatastoreException {
        return CONVERTER.toV4AllocateIdsResponse(datastore.allocateIds(
            CONVERTER.toV1AllocateIdsRequest(v4Request).build())).build();
      }
    });
  }

  @Override
  public Future<CommitResponse> commit(final CommitRequest v4Request) {
    return makeCall(new Callable<CommitResponse>() {
      @Override
      public CommitResponse call() throws DatastoreException {
        return CONVERTER.toV4CommitResponse(datastore.commit(
            CONVERTER.toV1CommitRequest(v4Request).build()), v4Request).build();
      }
    });
  }

  @Override
  public Future<CommitResponse> rawCommit(byte[] v4Request) {
    throw new UnsupportedOperationException();
  }

  private static <T extends Message> Future<T> makeCall(final Callable<T> request) {
    return executor.submit(new Callable<T>() {
      @Override
      public T call() throws Exception {
        try {
          return request.call();
        } catch (DatastoreException e) {
          throw extractException(e.getMessage(), e.getCode());
        }
      }
    });
  }

  private static final Pattern reasonPattern = Pattern.compile("\"reason\": \"(.*)\",?\\n");
  private static final Pattern messagePattern = Pattern.compile("\"message\": \"(.*)\",?\\n");

  /**
   * Convert the exception thrown by Cloud Datastore to version comparable
   * to the Exceptions thrown by ApiProxy.
   */
  protected static Exception extractException(String rawMessage, int httpCode) {
    Matcher msgMatcher = messagePattern.matcher(rawMessage);
    String message = msgMatcher.find() ? msgMatcher.group(1) : "[" + rawMessage + "]\n";
    switch (httpCode) {
      case HttpServletResponse.SC_BAD_REQUEST:
        return new IllegalArgumentException(message);
      case HttpServletResponse.SC_FORBIDDEN:
        Matcher reasonMatch = reasonPattern.matcher(rawMessage);
        if (reasonMatch.find() && reasonMatch.group(1).equals("DEADLINE_EXCEEDED")) {
          return new DatastoreTimeoutException(message);
        } else {
          return new IllegalStateException(message);
        }
      case HttpServletResponse.SC_PRECONDITION_FAILED:
        return new DatastoreNeedIndexException(message);
      case HttpServletResponse.SC_CONFLICT:
        return new ConcurrentModificationException(message);
      case HttpServletResponse.SC_SERVICE_UNAVAILABLE:
        return new IllegalStateException(message);
      case HttpServletResponse.SC_INTERNAL_SERVER_ERROR:
        return new DatastoreFailureException(message);
      case HttpServletResponse.SC_PAYMENT_REQUIRED:
      default:
        return new RuntimeException(message);
    }
  }

  private static DatastoreOptions getDatastoreOptions()
      throws GeneralSecurityException, IOException {
    DatastoreOptions.Builder options = new DatastoreOptions.Builder();
    options.dataset(EnvProxy.getenv("DATASTORE_DATASET"));
    options.host(EnvProxy.getenv("DATASTORE_HOST"));

    String serviceAccount = EnvProxy.getenv("DATASTORE_SERVICE_ACCOUNT");
    String privateKeyFile = EnvProxy.getenv("DATASTORE_PRIVATE_KEY_FILE");
    Credential credential;
    if (Boolean.valueOf(EnvProxy.getenv("__DATASTORE_USE_STUB_CREDENTIAL_FOR_TEST"))) {
      credential = null;
    } else if (serviceAccount != null && privateKeyFile != null) {
      credential = getServiceAccountCredential(serviceAccount, privateKeyFile);
    } else {
      credential = getComputeEngineCredential();
    }
    options.credential(credential);

    final String versionOverrideForTest = EnvProxy.getenv("__DATASTORE_VERSION_OVERRIDE_FOR_TEST");
    if (versionOverrideForTest != null) {
      options.initializer(new HttpRequestInitializer() {
        @Override
        public void initialize(HttpRequest request) throws IOException {
          request.getUrl().setRawPath(
              request.getUrl().getRawPath().replaceFirst(
                  DatastoreFactory.VERSION, versionOverrideForTest));
        }
      });
    }

    return options.build();
  }

  private static Credential getServiceAccountCredential(String account, String privateKeyFile)
      throws GeneralSecurityException, IOException {
    return new GoogleCredential.Builder()
        .setTransport(GoogleNetHttpTransport.newTrustedTransport())
        .setJsonFactory(new JacksonFactory())
        .setServiceAccountId(account)
        .setServiceAccountScopes(DatastoreOptions.SCOPES)
        .setServiceAccountPrivateKeyFromP12File(new File(privateKeyFile))
        .build();
  }

  private static Credential getComputeEngineCredential()
      throws GeneralSecurityException, IOException {
    NetHttpTransport transport = GoogleNetHttpTransport.newTrustedTransport();
    try {
      ComputeCredential credential = new ComputeCredential(transport, new JacksonFactory());
      credential.refreshToken();
      return credential;
    } catch (IOException e) {
      return null;
    }
  }

  /**
   * Make sure that the API proxy has been configured. If it's already
   * configured (e.g. because the Remote API has been installed or the factory
   * has already been used), do nothing. Otherwise, install a stub environment
   * and delegate.
   */
  private static synchronized void ensureApiProxyIsConfigured(DatastoreOptions options) {
    boolean hasEnvironmentOrFactory = (ApiProxy.getCurrentEnvironment() != null);
    boolean hasDelegate = (ApiProxy.getDelegate() != null);

    if (hasEnvironmentOrFactory && hasDelegate) {
      return;
    }

    if (hasEnvironmentOrFactory) {
      throw new IllegalStateException(
          "An ApiProxy.Environment or ApiProxy.EnvironmentFactory was already installed. "
          + "Cannot use Cloud Datastore.");
    } else if (hasDelegate) {
      throw new IllegalStateException(
          "An ApiProxy.Delegate was already installed. Cannot use Cloud Datastore.");
    }

    ApiProxy.setEnvironmentFactory(
        new StubApiProxyEnvironmentFactory(getFullAppId(options)));
    ApiProxy.setDelegate(new StubApiProxyDelegate());
  }

  /**
   * Attempt to determine the full app id. This is only necessary if the client
   * did not install the Remote API (which will determine it automatically).
   * <p>
   * By default, take the dataset from the provided {@link DatastoreOptions} and
   * prepend {@code s~}. Apps for which this is incorrect (e.g. apps running in
   * Europe) can specify the full app id via the {@code _DATASTORE_FULL_DATASET}
   * environment variable.
   */
  private static String getFullAppId(DatastoreOptions options) {
    String fullDataset = EnvProxy.getenv("_DATASTORE_FULL_DATASET");
    if (fullDataset != null) {
      return fullDataset;
    } else if (options.getHost().startsWith("http://localhost")) {
      return options.getDataset();
    }
    return "s~" + options.getDataset();
  }

  /**
   * A {@link Delegate} that throws {@link UnsupportedOperationException} for
   * all methods.
   */
  static class StubApiProxyDelegate implements Delegate<Environment> {
    private static final String UNSUPPORTED_API_PATTERN =
        "Calls to %s.%s are not supported under this configuration, only "
        + "calls to Cloud Datastore. To use other APIs, first install the "
        + "Remote API.";

    @Override
    public byte[] makeSyncCall(Environment environment, String packageName,
        String methodName, byte[] request) throws ApiProxyException {
      throw new UnsupportedOperationException(
          String.format(UNSUPPORTED_API_PATTERN, packageName, methodName));
    }

    @Override
    public Future<byte[]> makeAsyncCall(Environment environment, String packageName,
        String methodName, byte[] request, ApiConfig apiConfig) {
      throw new UnsupportedOperationException(
          String.format(UNSUPPORTED_API_PATTERN, packageName, methodName));
    }

    @Override
    public void log(Environment environment, LogRecord record) {
      throw new UnsupportedOperationException();
    }

    @Override
    public void flushLogs(Environment environment) {
      throw new UnsupportedOperationException();
    }

    @Override
    public List<Thread> getRequestThreads(Environment environment) {
      throw new UnsupportedOperationException();
    }
  }

  /**
   * An {@link EnvironmentFactory} that builds {@link StubApiProxyEnvironment}s.
   */
  static class StubApiProxyEnvironmentFactory implements EnvironmentFactory {
    private final String appId;

    public StubApiProxyEnvironmentFactory(String appId) {
      this.appId = appId;
    }

    @Override
    public Environment newEnvironment() {
      return new StubApiProxyEnvironment(appId);
    }
  }

  /**
   * An {@link Environment} that supports the minimal subset of features needed
   * to run code from the datastore package outside of App Engine. All other
   * methods throw {@link UnsupportedOperationException}.
   */
  static class StubApiProxyEnvironment implements Environment {
    private final Map<String, Object> attributes;
    private final String appId;

    public StubApiProxyEnvironment(String appId) {
      this.attributes = new HashMap<>();
      this.appId = appId;
    }

    @Override
    public boolean isLoggedIn() {
      throw new UnsupportedOperationException();
    }

    @Override
    public boolean isAdmin() {
      throw new UnsupportedOperationException();
    }

    @Override
    public String getVersionId() {
      throw new UnsupportedOperationException();
    }

    @Deprecated
    @Override
    public String getRequestNamespace() {
      throw new UnsupportedOperationException();
    }

    @Override
    public long getRemainingMillis() {
      throw new UnsupportedOperationException();
    }

    @Override
    public String getModuleId() {
      throw new UnsupportedOperationException();
    }

    @Override
    public String getEmail() {
      throw new UnsupportedOperationException();
    }

    @Override
    public String getAuthDomain() {
      throw new UnsupportedOperationException();
    }

    @Override
    public Map<String, Object> getAttributes() {
      return attributes;
    }

    @Override
    public String getAppId() {
      return appId;
    }
  }
}
