package com.google.appengine.api.search.checkers;

import com.google.apphosting.api.AppEngineInternal;

/**
 * Provides checks for faceted search related query options.
 *
 */
@AppEngineInternal
public class FacetQueryChecker {

  private static void checkMaximum(int number, int max, String context) {
    Preconditions.checkArgument(number > 0, "%s should be positive", context);
    Preconditions.checkArgument(number <= max,
        "%s must be less than or equal to %s", context, max);
  }

  /**
   * Checks that a discovery limit is valid. The value must
   * be greater than 0 and less than {@link SearchApiLimits#FACET_MAXIMUM_DISCOVERY_LIMIT}.
   *
   * @param value the discovery limit to check
   * @return the discovery limit
   * @throws IllegalArgumentException if the discovery limit is less than 1 or
   * greater than {@literal SearchApiLimits#FACET_MAXIMUM_DISCOVERY_LIMIT}.
   */
  public static int checkDiscoveryLimit(int value) {
    checkMaximum(value, SearchApiLimits.FACET_MAXIMUM_DISCOVERY_LIMIT,
        "Facet discovery limit");
    return value;
  }

  /**
   * Checks that a value constraint is valid. The Value length must
   * be greater than 0 and less than {@link SearchApiLimits#FACET_MAXIMUM_VALUE_LENGTH}.
   *
   * @param value the value constraint to check
   * @return the value constraint
   * @throws IllegalArgumentException if the Value length is less than 1 or
   * greater than {@literal SearchApiLimits#FACET_MAXIMUM_VALUE_LENGTH}.
   */
  public static String checkFacetValue(String value) {
    return FacetChecker.checkAtom(value);
  }

  /**
   * Checks that a facet depth option is valid. The facet depth must
   * be greater than 0 and less than {@link SearchApiLimits#FACET_MAXIMUM_DEPTH}.
   *
   * @param value the facet depth option to check
   * @return the facet depth
   * @throws IllegalArgumentException if the facet depth option is less than 1 or
   * greater than {@literal SearchApiLimits#FACET_MAXIMUM_DEPTH}.
   */
  public static Integer checkDepth(Integer value) {
    if (value != null) {
      checkMaximum(value, SearchApiLimits.FACET_MAXIMUM_DEPTH, "Facet depth option");
    }
    return value;
  }

  /**
   * Checks whether discovery value limit option is valid. The discovery value limit must
   * be greater than 0 and less than {@link SearchApiLimits#FACET_MAXIMUM_VALUE_LIMIT}.
   *
   * @param value the discovery value limit to check
   * @return the discovery value limit
   * @throws IllegalArgumentException if the discovery value limit is less than 1 or
   * greater than {@literal SearchApiLimits#FACET_MAXIMUM_VALUE_LIMIT}.
   */
  public static Integer checkDiscoveryValueLimit(Integer value) {
    if (value != null) {
      checkMaximum(value, SearchApiLimits.FACET_MAXIMUM_VALUE_LIMIT,
          "Facet discovery value limit");
    }
    return value;
  }

  /**
   * Checks whether a value limit option is valid. The value limit must
   * be greater than 0 and less than {@link SearchApiLimits#FACET_MAXIMUM_VALUE_LIMIT}.
   *
   * @param value the value limit to check
   * @return the value limit
   * @throws IllegalArgumentException if the value limit is less than 1 or
   * greater than {@literal SearchApiLimits#FACET_MAXIMUM_VALUE_LIMIT}.
   */
  public static Integer checkValueLimit(Integer value) {
    if (value != null) {
      checkMaximum(value, SearchApiLimits.FACET_MAXIMUM_VALUE_LIMIT, "Facet value limit");
    }
    return value;
  }

  private FacetQueryChecker() {
  }
}

